// $Id: CControlRoot.cpp,v 1.6 2007/03/03 14:45:48 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CControlRoot.hpp"
using Exponent::GUI::Controls::CControlRoot;

//	===========================================================================
CControlRoot::CControlRoot(IWindow *parent) 
			: m_subMouseEvent(parent->getMouse(), CMouseEvent::e_mouseMove)
			, m_controlArray(NULL)
			, m_lockedControl(NULL)
			, m_lastControl(NULL)
			, m_controlRootBackgroundImage(NULL)
			, m_parentWindow(NULL)
			, m_originalPosition(0, 0)
			, m_isVSTAttatched(false)
{
	// Store the parent
	this->setParentWindow(parent);

	// Create the array we will fill
	m_controlArray = new TCountedPointerArray<IControl>;

	// Null our pointers
	NULL_POINTER(m_lockedControl);
	NULL_POINTER(m_lastControl);
	NULL_POINTER(m_controlRootBackgroundImage);

	m_isVSTAttatched = false;
}

//	===========================================================================
CControlRoot::~CControlRoot()
{
	FREE_POINTER(m_controlArray);
	FORGET_COUNTED_OBJECT(m_controlRootBackgroundImage);
}

//	===========================================================================
void CControlRoot::lockControl(IControl *control)
{
	m_lockedControl = control;
}

//	===========================================================================
void CControlRoot::unlockControl()
{
	NULL_POINTER(m_lockedControl);
}

//	===========================================================================
bool CControlRoot::controlIsLocked() const
{
	return (m_lockedControl != NULL);
}

//	===========================================================================
IControl *CControlRoot::getLockedControl() const
{
	return m_lockedControl;
}

//	===========================================================================
void CControlRoot::setParentWindow(IWindow *parent)
{
	if (parent == NULL)
	{
		throw CException("Parent window is NULL", "CControlRoot::CControlRoot(IWindow *)");
	}
	m_parentWindow = parent;
}

//	===========================================================================
IWindow *CControlRoot::getParentWindow() const
{
	return m_parentWindow;
}

//	===========================================================================
CPoint CControlRoot::getWindowOffset()
{
	return CPoint(0, 0);
}

//	===========================================================================
IControl *CControlRoot::getControlAtPoint(const CPoint &point)
{
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		IControl *control = m_controlArray->elementAtIndex(i);
		if (control && control->getArea().pointIsInside(point))
		{
			return control;
		}
	}
	return NULL;
}

//	===========================================================================
IControl *CControlRoot::getMouseEnabledControlAtPoint(const CPoint &point)
{
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		IControl *control = m_controlArray->elementAtIndex(i);
		if (control && control->getArea().pointIsInside(point) && control->isMouseEnabled())
		{
			return control;
		}
	}
	return NULL;
}

//	===========================================================================
IControl *CControlRoot::getDropEnabledControlAtPoint(const CPoint &point)
{
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		IControl *control = m_controlArray->elementAtIndex(i);
		if (control && control->getArea().pointIsInside(point) && control->isDropFileEnabled())
		{
			return control;
		}
	}
	return NULL;
}

//	===========================================================================
IControl *CControlRoot::getKeyEnabledControlAtPoint(const CPoint &point)
{
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		IControl *control = m_controlArray->elementAtIndex(i);
		if (control && control->getArea().pointIsInside(point) && control->isKeyEnabled())
		{
			return control;
		}
	}
	return NULL;
}

//	===========================================================================
void CControlRoot::addControl(IControl *control)
{
	if (!m_controlArray->isPointerInArray(control))
	{
		m_controlArray->addElement(control);
	}
	else
	{
		// Move the control
	}
}

//	===========================================================================
void CControlRoot::removeControl(IControl *control)
{
	m_controlArray->deletePointerAtIndex(this->getControlIndex(control));
	//m_controlArray->releasePointerAtIndex(this->getControlIndex(control));
}

//	===========================================================================
void CControlRoot::clearControls()
{
	m_controlArray->clearArray();
}

//	===========================================================================
IControl *CControlRoot::getControlAtIndex(const long index, bool isIndex)
{
	if (isIndex)
	{
		return m_controlArray->elementAtIndex(index);
	}
	else
	{
		for (long i = 0; i < m_controlArray->getArraySize(); i++)
		{
			IControl *control = m_controlArray->elementAtIndex(i);
			if (control && control->getUniqueId() == index)
			{
				return control;
			}
		}
	}
	return NULL;
}

//	===========================================================================
long CControlRoot::getControlIndex(IControl *control)
{
	return m_controlArray->getIndexOfPointer(control);
}

//	===========================================================================
long CControlRoot::getNumberOfControls()
{
	return m_controlArray->getArraySize();
}

//	===========================================================================
long CControlRoot::getNumberOfInsertedControls()
{
	return m_controlArray->getInsertIndex();
}

//	===========================================================================
bool CControlRoot::isControlAttatched(IControl *control)
{
	return m_controlArray->isPointerInArray(control);
}

//	===========================================================================
void CControlRoot::drawRootControl(CGraphics &graphics)
{
	// ------------------------------------------------
	// Some information about drawing with the Exponent control system
	// 1. The graphics update area is the total area that will be blitted back
	//	  at the end of the drawing stage. For each control it is set to be relative 
	//    to their top left, as it is for us. If a control is inside the update
	//    area it will be drawn
	// 2. Control expect to draw based on their top left being 0, 0. Therefore
	//	  we need to offset the graphics position
	// 3. Controls expect that the clipping area is set to their actual onscreen
	//    position
	// 4. The absolute root control is the only place that you should use absolute
	//	  rectangle positions for controls. Otherwise you should always base
	//	  the position on the clipping rect, plus any other modifiers that
	//	  you wish to apply
	// ------------------------------------------------
	
	//CTrace::trace("Root level draw");

	// Store the update region
	const CRect update		   = graphics.getUpdateArea();
	const CPoint drawingOffset = graphics.getDrawingAreaOffset();
	CRect currentClip;
	graphics.getClippingRegion().getCurrentClipRegion(currentClip);
	
	// Draw the background colour...
	#ifndef WIN32
	if (m_parentWindow)
	{
		graphics.getClippingRegion().setCurrentClipRegion(update);
		CColour theColour = m_parentWindow->getWindowAttributes()->getBackgroundColour();
		graphics.getMutableBrush()->setColour(CAlphaColour(theColour.getRed(), theColour.getGreen(), theColour.getBlue(), 255));
		graphics.fillRectangle(update);
		graphics.getClippingRegion().clearClipRegion();
	}
	#endif

	// Draw the background image, if one is assigned
	if (m_controlRootBackgroundImage)
	{
		graphics.getClippingRegion().setCurrentClipRegion(update);
		graphics.drawImage(m_controlRootBackgroundImage, update, update);
		graphics.getClippingRegion().clearClipRegion();
	}
	
	// Now we loop through each control, updating sizes as necessary
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		// Get the control
		IControl *control = m_controlArray->elementAtIndex(i);

		// Check if its absolute area is inside our update area
		if (control && update.rectanglesIntersect(control->getAbsoluteRect()))
		{
			// Setup a new clipping region
			graphics.getClippingRegion().setCurrentClipRegion(control->getAbsoluteRect());

#ifdef DEBUG_CLIPPING
			graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_ORANGE);
			graphics.drawRectangle(control->getAbsoluteRect());
#endif

			// Now reoffset the graphics area so that controls draw from 0, 0 relative to themselves
			graphics.setDrawingAreaOffset(control->getAbsoluteRect().getOrigin());

			// If the control is entirely within the update area, then its update area is its area
			if (update.rectIsInside(control->getAbsoluteRect()))
			{
				graphics.setUpdateArea(CRect(0, 0, control->getAbsoluteRect().getWidth(), control->getAbsoluteRect().getHeight()));
			}
			else
			{
				// Otherwise its partially inside
				// In this case the rectangle will be the intersection area between the two controls
				CRect intersection;
				CRect::getIntersectionArea(update, control->getAbsoluteRect(), intersection);
				
				// This position is still a fixed onscreen position - we want to make it relative for the control
				intersection.offset(CPoint(-control->getArea().getLeft(), -control->getArea().getTop()));

				// Now we can store that as the update area for this control
				graphics.setUpdateArea(intersection);
			}

			// Draw 
			control->drawControl(graphics);

			// Now remove the offset from the graphics area so that future controls draw in the correct place....
			graphics.setDrawingAreaOffset(drawingOffset);

			// Reset the clipping region
			graphics.getClippingRegion().setCurrentClipRegion(currentClip);
		}
	}

	// Reset the drawing and update areas back to what they were
	graphics.setDrawingAreaOffset(drawingOffset);
	graphics.setUpdateArea(update);
}

//	===========================================================================
void CControlRoot::updateControl(IControl *control)
{
	//CTrace::trace("Updating the control");
	m_parentWindow->redrawWindow(control->getAbsoluteRect());

}

//	===========================================================================
void CControlRoot::updateControl(const long index, bool isIndex)
{
	IControl *control = this->getControlAtIndex(index, isIndex);
	if (control)
	{
		this->updateControl(control);
	}
}

//	===========================================================================
void CControlRoot::updateArea(const CRect &area)
{
	m_parentWindow->redrawWindow(area);
}

//	===========================================================================
void CControlRoot::getGlobalCoordinatesOfControl(IControl *control, CPoint &point)
{
	if (control)
	{
		CPoint globalLocation; 
		#ifdef WIN32
			/*if (!m_isVSTAttatched)
			{
				globalLocation = m_parentWindow->getWindowPosition();
			}
			else
			{*/
				CRect area;
				CWindowTools::getGlobalContentAreaCoordinates(m_parentWindow->getMutableWindowHandle(), area);
				globalLocation = area.getOrigin();
			//}
		#else
			CRect area;
			CWindowTools::getWindowArea(m_parentWindow->getMutableWindowHandle(), area);
			globalLocation = area.getOrigin();
		#endif
		const CRect absoluteArea = control->getAbsoluteRect();
		#ifdef WIN32
			point.setPoint(globalLocation.getXPosition() + absoluteArea.getLeft() - 1, globalLocation.getYPosition() + absoluteArea.getTop() + absoluteArea.getHeight());
		#else
			//if (!m_isVSTAttatched)
			//{
			//	point.setPoint(globalLocation.getXPosition() + absoluteArea.getLeft() - 1, globalLocation.getYPosition() + absoluteArea.getBottom());
			//}
			//else
			{
				// Get the size of the widnow, and position
				Rect windowStructureRgn;
				GetWindowBounds(m_parentWindow->getMutableWindowHandle()->m_windowHandle, kWindowStructureRgn, &windowStructureRgn);
				CDimension dim(abs(windowStructureRgn.left - windowStructureRgn.right), abs(windowStructureRgn.top - windowStructureRgn.bottom));
				
				// Get the size of the titlebar
				Rect titleBar;
				GetWindowBounds(m_parentWindow->getMutableWindowHandle()->m_windowHandle, kWindowTitleBarRgn, &titleBar);
				
				const long size = titleBar.bottom - titleBar.top;
				point.setPoint(globalLocation.getXPosition() + absoluteArea.getLeft() - 1, globalLocation.getYPosition() + absoluteArea.getBottom() + size);
			}
		#endif
	}
}

//	===========================================================================
void CControlRoot::getWindowCoordinatesOfControl(IControl *control, CPoint &point)
{
	// Check if the control is valid
	if (control)
	{
		// If so we offset it to the controls position
		point.offset(control->getArea().getOrigin());
	}
}

//	===========================================================================
void CControlRoot::setBackgroundImage(IImage *image)
{
	EXCHANGE_COUNTED_OBJECTS(m_controlRootBackgroundImage, image);
}

//	===========================================================================
void CControlRoot::startToolTipTimer()
{
	// We need to setup a timer here
}

//	===========================================================================
void CControlRoot::stopToolTipTimer() 
{
	// We need to setup a timer here
}

//	===========================================================================
IControl *CControlRoot::preProcessMouseEvent(CMouseEvent &event)
{
	// Copy the original position for later
	m_originalPosition = event.getMousePosition();

	// Determine if we have a control
	IControl *theControl = NULL;

	// Check if we have a valid control
	if (m_lockedControl == NULL)
	{
		// Get the control that they are mousing in
		theControl = this->getMouseEnabledControlAtPoint(m_originalPosition);

		// If we didnt get a control, we can return, nothing else to do
		if (theControl == NULL)
		{
			return NULL;
		}
	}
	else
	{
		theControl = m_lockedControl;
	}

	// Setup the sub event
	m_subMouseEvent.setModifiers(event.isCtrlDown(), event.isShiftDown(), event.isAltDown());

	// Set the sub event to have the correct position
	m_subMouseEvent.getMutableMouse()->setPosition(CPoint(m_originalPosition.getXPosition() - theControl->getArea().getLeft(), m_originalPosition.getYPosition() - theControl->getArea().getTop()));

	// return the control (which must have postProcessMouseEvent called on it)
	return theControl;
}

//	===========================================================================
void CControlRoot::postProcessMouseEvent(IControl *theControl)
{
	// reset the position
	m_subMouseEvent.getMutableMouse()->setPosition(m_originalPosition);

	// Store the control
	m_lastControl = theControl;
}

//	===========================================================================
void CControlRoot::handleLeftButtonDown(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleLeftButtonDown(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleLeftButtonUp(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleLeftButtonUp(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleRightButtonDown(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleRightButtonDown(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleRightButtonUp(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleRightButtonUp(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleDoubleClick(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleDoubleClick(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleMouseScroll(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}
	
	// Store the scoll position
	m_subMouseEvent.setWheelMovementAmount(event.getWheelMovementAmount());

	// Handle the mouse event
	theControl->handleMouseScroll(m_subMouseEvent);	

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleMouseMovement(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleMouseMovement(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
void CControlRoot::handleMouseLeavingArea(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	theControl->handleMouseLeavingArea(m_subMouseEvent);

	// Process the changes
	this->postProcessMouseEvent(theControl);
}

//	===========================================================================
bool CControlRoot::handleKeyDown(const CKeyboardEvent &event) 
{ 
	if (m_lockedControl == NULL)
	{
		if (m_lastControl == NULL)
		{
			return false;
		}
		return m_lastControl->handleKeyDown(event);
	}
	else
	{
		return m_lockedControl->handleKeyDown(event);
	}
	return false;
}

//	===========================================================================
bool CControlRoot::handleKeyUp(const CKeyboardEvent &event)  
{
	if (m_lockedControl == NULL)
	{
		if (m_lastControl == NULL)
		{
			return false;
		}
		return m_lastControl->handleKeyUp(event);
	}
	else
	{
		return m_lockedControl->handleKeyUp(event);
	}
	return false;
}

//	===========================================================================
void CControlRoot::handleFileDrop(const CDropEvent &event) 
{ 
	// Get the control
	IControl *theControl = this->getDropEnabledControlAtPoint(event.getDropFilePosition());

	if (theControl == NULL)
	{
		return;
	}

	// Process the message
	theControl->handleFileDrop(event);
}
